<?php

namespace App\Http\Controllers\Api;

use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{

    /**
     * @OA\Post(
     *   path="/api/auth/register",
     *   tags={"Auth - API"},
     *   summary="Store New User",
     *   description="It will return an access token",
     *   operationId="storeUser",       
     *   @OA\RequestBody(
     *       required=true,
     *       @OA\MediaType(
     *           mediaType="application/json",
     *           @OA\Schema(
     *               @OA\Property(property="name", type="string", example="John Doe"),
     *               @OA\Property(property="email", type="string", example="john.doe@example.com"),
     *               @OA\Property(property="password", type="string", example="password123"),
     *           )
     *       )
     *   ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     *
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function createUser(Request $request)
    {
        // Validate request data
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|unique:users|max:255',
            'password' => 'required|min:10',
        ]);
        // Return errors if validation error occur.
        if ($validator->fails()) {
            $errors = $validator->errors();
            return response()->json([
                'error' => $errors
            ], 400);
        }
        // Check if validation pass then create user and auth token. Return the auth token
        if ($validator->passes()) {
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password)
            ]);
            $token = $user->createToken('auth_token')->plainTextToken;
        
            return response()->json([
                'access_token' => $token,
                'token_type' => 'Bearer',
            ]);
        }
        
    }

    
    /**
     * @OA\Post(
     *   path="/api/auth/login",
     *   tags={"Auth - API"},
     *   summary="To login the user",
     *   description="It will return an access token",
     *   operationId="login user",       
     *   @OA\RequestBody(
     *       required=true,
     *       @OA\MediaType(
     *           mediaType="application/json",
     *           @OA\Schema(
     *               @OA\Property(property="email", type="string", example="john.doe@example.com"),
     *               @OA\Property(property="password", type="string", example="password123"),
     *           )
     *       )
     *   ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     *
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function loginUser(Request $request)
    {
        if (!Auth::attempt($request->only('email', 'password'))) {
            return response()->json([
                'message' => 'Invalid login details'
            ], 401);
        }
        $user = User::where('email', $request['email'])->firstOrFail();
        $token = $user->createToken('auth_token')->plainTextToken;
        return response()->json([
            'access_token' => $token,
            'token_type' => 'Bearer',
        ]);
    }

    /**
     * @OA\Post(
     *   path="/api/logout",
     *   tags={"Auth - API"},
     *   summary="To logout the user",
     *   description="It will return an access token",
     *   operationId="logout user",       
     *   security={{"bearerAuth":{}}},
     * 
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        auth()->user()->tokens()->delete();
        return response()->json([
            'message' => 'user logged out'
        ]);
    }

   /**
     * @OA\Get(
     *   path="/api/testing",
     *   tags={"Auth - API"},
     *   summary="To get the user",
     *   description="It will return an access token",
     *   operationId="get user",
     *   security={{"bearerAuth":{}}},
     * 
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     */
    public function testing(Request $request)
    {
        
        return response()->json(['success' => 'api auth successfully']);
    }



}
